<?php
include 'config.php';
include 'session.php';

// Handle POST requests (Add/Update)
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : 0;
    $nama = isset($_POST['nama']) ? trim($_POST['nama']) : '';
    $kontak = isset($_POST['kontak']) ? trim($_POST['kontak']) : '';
    $alamat = isset($_POST['alamat']) ? trim($_POST['alamat']) : '';
    
    // Validasi input
    if (empty($nama)) {
        $_SESSION['error_message'] = "Nama supplier tidak boleh kosong!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    // Validasi panjang nama
    if (strlen($nama) > 100) {
        $_SESSION['error_message'] = "Nama supplier maksimal 100 karakter!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    // Validasi kontak jika diisi
    if (!empty($kontak) && strlen($kontak) > 50) {
        $_SESSION['error_message'] = "Kontak maksimal 50 karakter!";
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
    
    try {
        if ($id > 0) {
            // UPDATE - cek apakah data exists
            $check_stmt = $conn->prepare("SELECT id FROM supplier WHERE id = ?");
            $check_stmt->bind_param("i", $id);
            $check_stmt->execute();
            $result = $check_stmt->get_result();
            
            if ($result->num_rows == 0) {
                $_SESSION['error_message'] = "Data yang akan diupdate tidak ditemukan!";
                $check_stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            }
            $check_stmt->close();
            
            // Cek duplikasi nama (kecuali untuk record yang sedang diedit)
            $dup_stmt = $conn->prepare("SELECT id FROM supplier WHERE nama = ? AND id != ?");
            $dup_stmt->bind_param("si", $nama, $id);
            $dup_stmt->execute();
            $dup_result = $dup_stmt->get_result();
            
            if ($dup_result->num_rows > 0) {
                $_SESSION['error_message'] = "Nama supplier '$nama' sudah ada!";
                $dup_stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            }
            $dup_stmt->close();
            
            // UPDATE
            $stmt = $conn->prepare("UPDATE supplier SET nama = ?, kontak = ?, alamat = ? WHERE id = ?");
            $stmt->bind_param("sssi", $nama, $kontak, $alamat, $id);
            $message = "Data berhasil diupdate!";
        } else {
            // INSERT - cek duplikasi nama
            $dup_stmt = $conn->prepare("SELECT id FROM supplier WHERE nama = ?");
            $dup_stmt->bind_param("s", $nama);
            $dup_stmt->execute();
            $dup_result = $dup_stmt->get_result();
            
            if ($dup_result->num_rows > 0) {
                $_SESSION['error_message'] = "Nama supplier '$nama' sudah ada!";
                $dup_stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            }
            $dup_stmt->close();
            
            // INSERT
            $stmt = $conn->prepare("INSERT INTO supplier (nama, kontak, alamat) VALUES (?, ?, ?)");
            $stmt->bind_param("sss", $nama, $kontak, $alamat);
            $message = "Data berhasil ditambahkan!";
        }
        
        if ($stmt->execute()) {
            $_SESSION['success_message'] = $message;
        } else {
            $_SESSION['error_message'] = "Gagal menyimpan data: " . $stmt->error;
        }
        $stmt->close();
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error: " . $e->getMessage();
        error_log("Master Supplier Error: " . $e->getMessage());
    }
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handle DELETE requests
if (isset($_GET['hapus']) && is_numeric($_GET['hapus'])) {
    $id = (int)$_GET['hapus'];
    
    try {
        // Cek apakah data exists
        $check_stmt = $conn->prepare("SELECT nama FROM supplier WHERE id = ?");
        $check_stmt->bind_param("i", $id);
        $check_stmt->execute();
        $result = $check_stmt->get_result();
        
        if ($result->num_rows == 0) {
            $_SESSION['error_message'] = "Data yang akan dihapus tidak ditemukan!";
        } else {
            $row = $result->fetch_assoc();
            
            // Cek apakah supplier sedang digunakan di tabel lain (jika ada)
            // Uncomment dan sesuaikan jika ada relasi ke tabel lain
            /*
            $usage_check = $conn->prepare("SELECT COUNT(*) as count FROM pembelian WHERE supplier_id = ?");
            $usage_check->bind_param("i", $id);
            $usage_check->execute();
            $usage_result = $usage_check->get_result();
            $usage_data = $usage_result->fetch_assoc();
            
            if ($usage_data['count'] > 0) {
                $_SESSION['error_message'] = "Supplier '{$row['nama']}' tidak dapat dihapus karena sedang digunakan!";
                $usage_check->close();
                $check_stmt->close();
                header("Location: " . $_SERVER['PHP_SELF']);
                exit;
            }
            $usage_check->close();
            */
            
            // DELETE
            $stmt = $conn->prepare("DELETE FROM supplier WHERE id = ?");
            $stmt->bind_param("i", $id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Supplier '{$row['nama']}' berhasil dihapus!";
            } else {
                $_SESSION['error_message'] = "Gagal menghapus data: " . $stmt->error;
            }
            $stmt->close();
        }
        $check_stmt->close();
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error saat menghapus: " . $e->getMessage();
        error_log("Master Supplier Delete Error: " . $e->getMessage());
    }
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Get all suppliers with error handling
try {
    $supplier = mysqli_query($conn, "SELECT * FROM supplier ORDER BY nama ASC");
    if (!$supplier) {
        throw new Exception("Error fetching data: " . mysqli_error($conn));
    }
    $totalSupplier = mysqli_num_rows($supplier);
} catch (Exception $e) {
    $_SESSION['error_message'] = "Error mengambil data: " . $e->getMessage();
    $totalSupplier = 0;
    $supplier = false;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Master Supplier</title>
    <link rel="icon" type="image/x-icon" href="/images/favicon.ico">
</head>
<body>

<!-- Include Menu Navigation -->
<?php 
// Set variabel untuk navbar
if (!isset($jabatan_relasi)) {
    $jabatan_relasi = $_SESSION['jabatan'] ?? '';
}
include 'navbar.php'; 
?>

<div class="container-fluid py-4">
    
    <!-- Page Header -->
    <div class="row mb-4">
        <div class="col">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h3 class="mb-1 text-dark">
                        <i class="fa fa-truck text-primary"></i> Master Supplier
                    </h3>
                    <p class="text-muted mb-0">Kelola data supplier perusahaan</p>
                </div>
                <div>
                    <span class="badge bg-primary fs-6">
                        <i class="fa fa-database"></i> <?= $totalSupplier ?> Supplier
                    </span>
                </div>
            </div>
        </div>
    </div>

    <!-- Alert Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fa fa-check-circle"></i> <?= htmlspecialchars($_SESSION['success_message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fa fa-exclamation-circle"></i> <?= htmlspecialchars($_SESSION['error_message']) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>

    <!-- Action Buttons & Search -->
    <div class="row mb-3">
        <div class="col-md-6">
            <button class="btn btn-success me-2" onclick="openModal()">
                <i class="fa fa-plus"></i> Tambah Supplier Baru
            </button>
            <button class="btn btn-outline-secondary" onclick="location.reload()">
                <i class="fa fa-refresh"></i> Refresh
            </button>
        </div>
        <div class="col-md-6">
            <div class="input-group">
                <span class="input-group-text">
                    <i class="fa fa-search"></i>
                </span>
                <input type="text" 
                       class="form-control" 
                       id="searchInput" 
                       placeholder="Cari supplier..." 
                       onkeyup="searchTable()">
                <button class="btn btn-outline-secondary" onclick="clearSearch()">
                    <i class="fa fa-times"></i>
                </button>
            </div>
        </div>
    </div>

    <!-- Data Table -->
    <div class="row">
        <div class="col">
            <div class="card shadow-sm">
                <div class="card-header bg-dark text-white">
                    <h5 class="mb-0">
                        <i class="fa fa-list"></i> Daftar Supplier
                    </h5>
                </div>
                <div class="card-body p-0">
                    <?php if ($supplier && $totalSupplier > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th width="60" class="text-center">#</th>
                                        <th>Nama Supplier</th>
                                        <th width="150">Kontak</th>
                                        <th width="200">Alamat</th>
                                        <th width="180" class="text-center">Aksi</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $no = 1;
                                    mysqli_data_seek($supplier, 0); // Reset pointer
                                    while ($row = mysqli_fetch_assoc($supplier)): 
                                    ?>
                                        <tr>
                                            <td class="text-center">
                                                <span class="badge bg-secondary"><?= $no++ ?></span>
                                            </td>
                                            <td>
                                                <strong><?= htmlspecialchars($row['nama']) ?></strong>
                                            </td>
                                            <td>
                                                <span class="text-muted">
                                                    <?= !empty($row['kontak']) ? htmlspecialchars($row['kontak']) : '<em class="text-secondary">-</em>' ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="text-muted small">
                                                    <?= !empty($row['alamat']) ? htmlspecialchars(substr($row['alamat'], 0, 50) . (strlen($row['alamat']) > 50 ? '...' : '')) : '<em class="text-secondary">-</em>' ?>
                                                </span>
                                            </td>
                                            <td class="text-center">
                                                <div class="btn-group" role="group">
                                                    <button class="btn btn-warning btn-sm" 
                                                            onclick='editModal(<?= json_encode($row, JSON_HEX_APOS | JSON_HEX_QUOT) ?>)' 
                                                            title="Edit Supplier">
                                                        <i class="fa fa-edit"></i> Edit
                                                    </button>
                                                    <button class="btn btn-danger btn-sm" 
                                                            onclick="confirmDelete(<?= (int)$row['id'] ?>, '<?= htmlspecialchars($row['nama'], ENT_QUOTES) ?>')"
                                                            title="Hapus Supplier">
                                                        <i class="fa fa-trash"></i> Hapus
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fa fa-search fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted" id="emptyMessage">
                                <?= $supplier === false ? 'Error mengambil data' : 'Belum ada data supplier' ?>
                            </h5>
                            <p class="text-muted" id="emptySubMessage">
                                <?= $supplier === false ? 'Silakan refresh halaman atau hubungi administrator' : 'Klik tombol "Tambah Supplier Baru" untuk menambah data pertama' ?>
                            </p>
                            <?php if ($supplier !== false): ?>
                                <button class="btn btn-success" onclick="openModal()" id="emptyAddBtn">
                                    <i class="fa fa-plus"></i> Tambah Sekarang
                                </button>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

</div>

<!-- Modal Add/Edit -->
<div class="modal fade" id="supplierModal" tabindex="-1" aria-labelledby="modalTitle" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <form method="post" class="modal-content" onsubmit="return validateForm()">
            <div class="modal-header">
                <h5 class="modal-title" id="modalTitle">
                    <i class="fa fa-plus"></i> Tambah Supplier
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="id" id="supplierId">
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="supplierNama" class="form-label">
                                <i class="fa fa-building"></i> Nama Supplier <span class="text-danger">*</span>
                            </label>
                            <input type="text" 
                                   class="form-control" 
                                   name="nama" 
                                   id="supplierNama" 
                                   placeholder="Masukkan nama supplier..." 
                                   required 
                                   maxlength="100"
                                   autocomplete="off">
                            <div class="form-text">Maksimal 100 karakter</div>
                            <div class="invalid-feedback" id="namaError"></div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label for="supplierKontak" class="form-label">
                                <i class="fa fa-phone"></i> Kontak
                            </label>
                            <input type="text" 
                                   class="form-control" 
                                   name="kontak" 
                                   id="supplierKontak" 
                                   placeholder="No. telepon / email..." 
                                   maxlength="50"
                                   autocomplete="off">
                            <div class="form-text">Maksimal 50 karakter</div>
                            <div class="invalid-feedback" id="kontakError"></div>
                        </div>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="supplierAlamat" class="form-label">
                        <i class="fa fa-map-marker-alt"></i> Alamat
                    </label>
                    <textarea class="form-control" 
                              name="alamat" 
                              id="supplierAlamat" 
                              rows="3"
                              placeholder="Masukkan alamat lengkap supplier..."
                              maxlength="500"></textarea>
                    <div class="form-text">Maksimal 500 karakter</div>
                    <div class="invalid-feedback" id="alamatError"></div>
                </div>
                
                <div class="alert alert-info" role="alert">
                    <i class="fa fa-info-circle"></i>
                    <strong>Catatan:</strong> Pastikan nama supplier unik. Field kontak dan alamat bersifat opsional.
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" class="btn btn-success" id="submitBtn">
                    <i class="fa fa-save"></i> Simpan
                </button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fa fa-times"></i> Batal
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Confirmation Modal -->
<div class="modal fade" id="confirmModal" tabindex="-1">
    <div class="modal-dialog modal-sm">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fa fa-exclamation-triangle text-warning"></i> Konfirmasi
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Yakin ingin menghapus supplier ini?</p>
                <p class="text-muted"><strong id="confirmItemName"></strong></p>
            </div>
            <div class="modal-footer">
                <a href="#" id="confirmDeleteBtn" class="btn btn-danger">
                    <i class="fa fa-trash"></i> Ya, Hapus
                </a>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
            </div>
        </div>
    </div>
</div>

<script>
// Function untuk validasi form
function validateForm() {
    const nama = document.getElementById('supplierNama');
    const kontak = document.getElementById('supplierKontak');
    const alamat = document.getElementById('supplierAlamat');
    const namaError = document.getElementById('namaError');
    const kontakError = document.getElementById('kontakError');
    const alamatError = document.getElementById('alamatError');
    const submitBtn = document.getElementById('submitBtn');
    
    // Reset validation
    nama.classList.remove('is-invalid');
    kontak.classList.remove('is-invalid');
    alamat.classList.remove('is-invalid');
    namaError.textContent = '';
    kontakError.textContent = '';
    alamatError.textContent = '';
    
    let isValid = true;
    
    // Validasi nama
    if (nama.value.trim() === '') {
        nama.classList.add('is-invalid');
        namaError.textContent = 'Nama supplier tidak boleh kosong!';
        nama.focus();
        isValid = false;
    } else if (nama.value.trim().length > 100) {
        nama.classList.add('is-invalid');
        namaError.textContent = 'Nama supplier maksimal 100 karakter!';
        nama.focus();
        isValid = false;
    }
    
    // Validasi kontak jika diisi
    if (kontak.value.trim() !== '' && kontak.value.trim().length > 50) {
        kontak.classList.add('is-invalid');
        kontakError.textContent = 'Kontak maksimal 50 karakter!';
        if (isValid) kontak.focus();
        isValid = false;
    }
    
    // Validasi alamat jika diisi
    if (alamat.value.trim() !== '' && alamat.value.trim().length > 500) {
        alamat.classList.add('is-invalid');
        alamatError.textContent = 'Alamat maksimal 500 karakter!';
        if (isValid) alamat.focus();
        isValid = false;
    }
    
    if (isValid) {
        // Disable submit button to prevent double submission
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Menyimpan...';
    }
    
    return isValid;
}

// Function untuk search table
function searchTable() {
    const input = document.getElementById('searchInput');
    const filter = input.value.toLowerCase();
    const table = document.querySelector('.table tbody');
    const rows = table ? table.getElementsByTagName('tr') : [];
    let visibleRows = 0;

    for (let i = 0; i < rows.length; i++) {
        const nameCell = rows[i].getElementsByTagName('td')[1]; // kolom nama supplier
        const kontakCell = rows[i].getElementsByTagName('td')[2]; // kolom kontak
        const alamatCell = rows[i].getElementsByTagName('td')[3]; // kolom alamat
        
        if (nameCell) {
            const nameValue = nameCell.textContent || nameCell.innerText;
            const kontakValue = kontakCell ? (kontakCell.textContent || kontakCell.innerText) : '';
            const alamatValue = alamatCell ? (alamatCell.textContent || alamatCell.innerText) : '';
            
            const combinedText = (nameValue + ' ' + kontakValue + ' ' + alamatValue).toLowerCase();
            
            if (combinedText.indexOf(filter) > -1) {
                rows[i].style.display = '';
                visibleRows++;
            } else {
                rows[i].style.display = 'none';
            }
        }
    }

    // Update empty state
    updateEmptyState(visibleRows, filter);
}

// Function untuk update empty state
function updateEmptyState(visibleRows, filter) {
    const emptyState = document.querySelector('.text-center.py-5');
    const tableContainer = document.querySelector('.table-responsive');
    
    if (visibleRows === 0 && filter !== '') {
        // Show "no results" message
        if (tableContainer) tableContainer.style.display = 'none';
        if (emptyState) {
            emptyState.style.display = 'block';
            const emptyMessage = document.getElementById('emptyMessage');
            const emptySubMessage = document.getElementById('emptySubMessage');
            const emptyAddBtn = document.getElementById('emptyAddBtn');
            
            if (emptyMessage) emptyMessage.textContent = 'Tidak ada hasil';
            if (emptySubMessage) emptySubMessage.textContent = `Tidak ditemukan supplier dengan kata kunci "${filter}"`;
            if (emptyAddBtn) emptyAddBtn.style.display = 'none';
        }
    } else if (visibleRows === 0 && filter === '') {
        // Show original empty state when no filter and no data
        if (tableContainer) tableContainer.style.display = 'none';
        if (emptyState) {
            emptyState.style.display = 'block';
            const emptyMessage = document.getElementById('emptyMessage');
            const emptySubMessage = document.getElementById('emptySubMessage');
            const emptyAddBtn = document.getElementById('emptyAddBtn');
            
            if (emptyMessage) emptyMessage.textContent = 'Belum ada data supplier';
            if (emptySubMessage) emptySubMessage.textContent = 'Klik tombol "Tambah Supplier Baru" untuk menambah data pertama';
            if (emptyAddBtn) emptyAddBtn.style.display = 'inline-block';
        }
    } else {
        // Show table when there are visible rows
        if (tableContainer) tableContainer.style.display = 'block';
        if (emptyState) emptyState.style.display = 'none';
    }
}

// Function untuk clear search
function clearSearch() {
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.value = '';
        searchTable();
        searchInput.focus();
    }
}

// Function untuk membuka modal tambah
function openModal() {
    const modalTitle = document.getElementById('modalTitle');
    const supplierId = document.getElementById('supplierId');
    const supplierNama = document.getElementById('supplierNama');
    const supplierKontak = document.getElementById('supplierKontak');
    const supplierAlamat = document.getElementById('supplierAlamat');
    const submitBtn = document.getElementById('submitBtn');
    
    if (modalTitle) modalTitle.innerHTML = '<i class="fa fa-plus"></i> Tambah Supplier';
    if (supplierId) supplierId.value = '';
    if (supplierNama) {
        supplierNama.value = '';
        supplierNama.classList.remove('is-invalid');
    }
    if (supplierKontak) {
        supplierKontak.value = '';
        supplierKontak.classList.remove('is-invalid');
    }
    if (supplierAlamat) {
        supplierAlamat.value = '';
        supplierAlamat.classList.remove('is-invalid');
    }
    if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fa fa-save"></i> Simpan';
    }
    
    const modal = document.getElementById('supplierModal');
    if (modal && typeof bootstrap !== 'undefined') {
        const bootstrapModal = new bootstrap.Modal(modal);
        bootstrapModal.show();
        
        // Focus on input after modal is shown
        modal.addEventListener('shown.bs.modal', function () {
            if (supplierNama) supplierNama.focus();
        }, { once: true });
    }
}

// Function untuk membuka modal edit
function editModal(data) {
    const modalTitle = document.getElementById('modalTitle');
    const supplierId = document.getElementById('supplierId');
    const supplierNama = document.getElementById('supplierNama');
    const supplierKontak = document.getElementById('supplierKontak');
    const supplierAlamat = document.getElementById('supplierAlamat');
    const submitBtn = document.getElementById('submitBtn');
    
    if (modalTitle) modalTitle.innerHTML = '<i class="fa fa-edit"></i> Edit Supplier';
    if (supplierId) supplierId.value = data.id;
    if (supplierNama) {
        supplierNama.value = data.nama;
        supplierNama.classList.remove('is-invalid');
    }
    if (supplierKontak) {
        supplierKontak.value = data.kontak || '';
        supplierKontak.classList.remove('is-invalid');
    }
    if (supplierAlamat) {
        supplierAlamat.value = data.alamat || '';
        supplierAlamat.classList.remove('is-invalid');
    }
    if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fa fa-save"></i> Update';
    }
    
    const modal = document.getElementById('supplierModal');
    if (modal && typeof bootstrap !== 'undefined') {
        const bootstrapModal = new bootstrap.Modal(modal);
        bootstrapModal.show();
        
        // Focus on input after modal is shown
        modal.addEventListener('shown.bs.modal', function () {
            if (supplierNama) {
                supplierNama.focus();
                supplierNama.select();
            }
        }, { once: true });
    }
}

// Function untuk konfirmasi hapus
function confirmDelete(id, nama) {
    const confirmItemName = document.getElementById('confirmItemName');
    const confirmDeleteBtn = document.getElementById('confirmDeleteBtn');
    
    if (confirmItemName) confirmItemName.textContent = nama;
    if (confirmDeleteBtn) confirmDeleteBtn.href = '?hapus=' + id;
    
    const modal = document.getElementById('confirmModal');
    if (modal && typeof bootstrap !== 'undefined') {
        new bootstrap.Modal(modal).show();
    }
}

// Auto-hide alerts after 5 seconds
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert-dismissible');
    alerts.forEach(function(alert) {
        setTimeout(function() {
            if (alert && alert.parentNode) {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            }
        }, 5000);
    });
    
    // Initialize search functionality
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        // Add event listener for real-time search
        searchInput.addEventListener('input', searchTable);
        searchInput.addEventListener('keyup', searchTable);
    }
    
    // Real-time validation
    const inputs = ['supplierNama', 'supplierKontak', 'supplierAlamat'];
    inputs.forEach(function(inputId) {
        const input = document.getElementById(inputId);
        if (input) {
            input.addEventListener('input', function() {
                if (this.classList.contains('is-invalid')) {
                    this.classList.remove('is-invalid');
                    const errorElement = document.getElementById(inputId.replace('supplier', '').toLowerCase() + 'Error');
                    if (errorElement) errorElement.textContent = '';
                }
            });
        }
    });
});

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    // Ctrl+Enter untuk submit form
    if (e.key === 'Enter' && e.ctrlKey) {
        const activeModal = document.querySelector('.modal.show');
        if (activeModal) {
            const submitBtn = activeModal.querySelector('button[type="submit"]');
            if (submitBtn && !submitBtn.disabled) {
                submitBtn.click();
            }
        }
    }
    
    // ESC key to clear search
    if (e.key === 'Escape') {
        const searchInput = document.getElementById('searchInput');
        if (searchInput && document.activeElement === searchInput && searchInput.value !== '') {
            clearSearch();
        }
    }
    
    // Ctrl+N untuk tambah baru
    if (e.key === 'n' && e.ctrlKey && !e.shiftKey && !e.altKey) {
        e.preventDefault();
        openModal();
    }
});
</script>

</body>
</html>