<?php
header('Content-Type: application/json');

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors for AJAX

// Include config
if (!file_exists('config.php')) {
    echo json_encode(['success' => false, 'message' => 'File config.php tidak ditemukan']);
    exit();
}

include 'config.php';

// Check connection
if (!isset($conn) || !$conn) {
    echo json_encode(['success' => false, 'message' => 'Database connection tidak tersedia']);
    exit();
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['action']) || $input['action'] !== 'create_tables') {
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    exit();
}

// Start transaction
mysqli_autocommit($conn, false);

try {
    // SQL to create meeting tables
    $create_tables = [
        'meeting' => "
            CREATE TABLE IF NOT EXISTS `meeting` (
              `id` int(11) NOT NULL AUTO_INCREMENT,
              `kode_meeting` varchar(50) NOT NULL DEFAULT '',
              `tanggal` date NOT NULL,
              `waktu_mulai` time DEFAULT NULL,
              `waktu_selesai` time DEFAULT NULL,
              `tempat` varchar(255) NOT NULL,
              `jenis_meeting` enum('internal','external','customer','supplier','planning','review') DEFAULT 'internal',
              `status` enum('planned','ongoing','completed','cancelled') DEFAULT 'planned',
              `pic_meeting` int(11) DEFAULT NULL,
              `customer_id` int(11) DEFAULT NULL,
              `supplier_id` int(11) DEFAULT NULL,
              `created_by` int(11) NOT NULL,
              `created_at` datetime NOT NULL DEFAULT current_timestamp(),
              `updated_at` datetime DEFAULT NULL ON UPDATE current_timestamp(),
              PRIMARY KEY (`id`),
              KEY `idx_tanggal` (`tanggal`),
              KEY `idx_status` (`status`),
              KEY `idx_jenis` (`jenis_meeting`),
              KEY `idx_created_by` (`created_by`)
            ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci
        ",
        
        'meeting_agenda' => "
            CREATE TABLE IF NOT EXISTS `meeting_agenda` (
              `id` int(11) NOT NULL AUTO_INCREMENT,
              `meeting_id` int(11) NOT NULL,
              `urutan` int(11) NOT NULL DEFAULT 1,
              `subject` varchar(255) NOT NULL,
              `detail_subject` text DEFAULT NULL,
              `status_agenda` enum('pending','discussed','completed','postponed') DEFAULT 'pending',
              `pic_agenda` int(11) DEFAULT NULL,
              `deadline` date DEFAULT NULL,
              `prioritas` enum('low','medium','high','urgent') DEFAULT 'medium',
              PRIMARY KEY (`id`),
              KEY `idx_meeting_agenda` (`meeting_id`),
              KEY `idx_urutan` (`urutan`)
            ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci
        ",
        
        'meeting_peserta' => "
            CREATE TABLE IF NOT EXISTS `meeting_peserta` (
              `id` int(11) NOT NULL AUTO_INCREMENT,
              `meeting_id` int(11) NOT NULL,
              `user_id` int(11) DEFAULT NULL,
              `nama_external` varchar(100) DEFAULT NULL,
              `jabatan_external` varchar(100) DEFAULT NULL,
              `perusahaan_external` varchar(100) DEFAULT NULL,
              `status_kehadiran` enum('planned','hadir','tidak_hadir','terlambat') DEFAULT 'planned',
              `keterangan` text DEFAULT NULL,
              PRIMARY KEY (`id`),
              KEY `idx_meeting_peserta` (`meeting_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci
        ",
        
        'meeting_action_items' => "
            CREATE TABLE IF NOT EXISTS `meeting_action_items` (
              `id` int(11) NOT NULL AUTO_INCREMENT,
              `meeting_id` int(11) NOT NULL,
              `agenda_id` int(11) DEFAULT NULL,
              `action_item` text NOT NULL,
              `pic_action` int(11) NOT NULL,
              `deadline` date DEFAULT NULL,
              `status_action` enum('open','in_progress','completed','cancelled','overdue') DEFAULT 'open',
              `progress_notes` text DEFAULT NULL,
              `completed_at` datetime DEFAULT NULL,
              `created_at` datetime NOT NULL DEFAULT current_timestamp(),
              `updated_at` datetime DEFAULT NULL ON UPDATE current_timestamp(),
              PRIMARY KEY (`id`),
              KEY `idx_meeting_action` (`meeting_id`),
              KEY `idx_agenda_action` (`agenda_id`),
              KEY `idx_status_deadline` (`status_action`, `deadline`)
            ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci
        ",
        
        'meeting_attachments' => "
            CREATE TABLE IF NOT EXISTS `meeting_attachments` (
              `id` int(11) NOT NULL AUTO_INCREMENT,
              `meeting_id` int(11) NOT NULL,
              `agenda_id` int(11) DEFAULT NULL,
              `file_name` varchar(255) NOT NULL,
              `file_path` varchar(500) NOT NULL,
              `file_type` varchar(50) NOT NULL,
              `file_size` bigint(20) DEFAULT NULL,
              `mime_type` varchar(100) DEFAULT NULL,
              `description` varchar(255) DEFAULT NULL,
              `uploaded_by` int(11) NOT NULL,
              `uploaded_at` datetime NOT NULL DEFAULT current_timestamp(),
              PRIMARY KEY (`id`),
              KEY `idx_meeting_attachment` (`meeting_id`),
              KEY `idx_agenda_attachment` (`agenda_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci
        ",
        
        'meeting_notes' => "
            CREATE TABLE IF NOT EXISTS `meeting_notes` (
              `id` int(11) NOT NULL AUTO_INCREMENT,
              `meeting_id` int(11) NOT NULL,
              `agenda_id` int(11) DEFAULT NULL,
              `catatan` text NOT NULL,
              `jenis_catatan` enum('general','decision','issue','information') DEFAULT 'general',
              `created_by` int(11) NOT NULL,
              `created_at` datetime NOT NULL DEFAULT current_timestamp(),
              `updated_at` datetime DEFAULT NULL ON UPDATE current_timestamp(),
              PRIMARY KEY (`id`),
              KEY `idx_meeting_notes` (`meeting_id`),
              KEY `idx_agenda_notes` (`agenda_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci
        "
    ];
    
    $created_tables = [];
    $errors = [];
    
    // Create each table
    foreach ($create_tables as $table_name => $sql) {
        if (mysqli_query($conn, $sql)) {
            $created_tables[] = $table_name;
        } else {
            $errors[] = "Error creating table $table_name: " . mysqli_error($conn);
        }
    }
    
    // Create trigger for auto-generating meeting code
    $trigger_sql = "
        CREATE TRIGGER `tr_meeting_generate_kode` BEFORE INSERT ON `meeting` FOR EACH ROW 
        BEGIN
            DECLARE next_num INT;
            DECLARE kode_baru VARCHAR(50);
            
            IF NEW.kode_meeting = '' OR NEW.kode_meeting IS NULL THEN
                SELECT COALESCE(MAX(CAST(SUBSTRING(kode_meeting, -4) AS UNSIGNED)), 0) + 1 
                INTO next_num
                FROM meeting 
                WHERE YEAR(tanggal) = YEAR(NEW.tanggal) AND kode_meeting LIKE CONCAT('MTG', YEAR(NEW.tanggal), '%');
                
                SET kode_baru = CONCAT('MTG', YEAR(NEW.tanggal), LPAD(next_num, 4, '0'));
                SET NEW.kode_meeting = kode_baru;
            END IF;
        END
    ";
    
    // Drop trigger if exists first
    mysqli_query($conn, "DROP TRIGGER IF EXISTS tr_meeting_generate_kode");
    if (mysqli_query($conn, $trigger_sql)) {
        $created_tables[] = 'trigger: tr_meeting_generate_kode';
    } else {
        $errors[] = "Error creating trigger: " . mysqli_error($conn);
    }
    
    // Create sample data
    $count_result = mysqli_query($conn, "SELECT COUNT(*) as count FROM meeting");
    if ($count_result) {
        $count_row = mysqli_fetch_assoc($count_result);
        if ($count_row['count'] == 0) {
            // Insert sample meeting
            $sample_sql = "INSERT INTO meeting (tanggal, tempat, jenis_meeting, status, created_by, kode_meeting) VALUES (CURDATE(), 'Ruang Rapat Utama', 'internal', 'planned', 1, '')";
            if (mysqli_query($conn, $sample_sql)) {
                $created_tables[] = 'sample data';
            } else {
                $errors[] = "Error creating sample data: " . mysqli_error($conn);
            }
        }
    }
    
    // Commit transaction
    mysqli_commit($conn);
    
    if (empty($errors)) {
        echo json_encode([
            'success' => true, 
            'message' => 'Meeting module berhasil disetup!',
            'created_tables' => $created_tables
        ]);
    } else {
        echo json_encode([
            'success' => true, 
            'message' => 'Setup selesai dengan beberapa warning',
            'created_tables' => $created_tables,
            'warnings' => $errors
        ]);
    }
    
} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($conn);
    
    echo json_encode([
        'success' => false, 
        'message' => 'Setup gagal: ' . $e->getMessage()
    ]);
}

// Check for mysqli errors
if (mysqli_error($conn)) {
    mysqli_rollback($conn);
    echo json_encode([
        'success' => false, 
        'message' => 'MySQL Error: ' . mysqli_error($conn)
    ]);
}
?>